### A Pluto.jl notebook ###
# v0.19.12

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
end

# ╔═╡ feb39810-13bf-4383-8184-9338111248cf
begin
	using PlotlyBase, HypertextLiteral, PlutoUI
	using CSV, DataFrames, PeriodicalDates, Dates
	#import PlotlyJS:savefig
end

# ╔═╡ 15112ba0-cddf-11eb-07a3-2bcd9257c8d0
md"""
# Week 5 - Monetary Policy & Aggregate Demand
# Exercises

**Macroeconomics, ISCTE-IUL**

"""

# ╔═╡ 3cdf6437-0705-410c-8b0b-e8fc657897ed
md"""
**Vivaldo Mendes, October 2022**
"""

# ╔═╡ 409f9114-3fe1-4abe-8beb-ab279e9031b2
md"""
### Packages
"""

# ╔═╡ 6937b9af-1f25-4b10-8820-86f5db2f84e9
begin
	function Base.show(io::IO, mimetype::MIME"text/html", plt::PlotlyBase.Plot)
       # Remove responsive flag on the plot as we handle responsibity via ResizeObeserver and leaving it on makes the div flickr during updates
	hasproperty(plt,:config) && plt.config.responsive && (plt.config.responsive = false)   
	show(io,mimetype, @htl("""
			<div>
			<script id=asdf>
			const {plotly} = await import("https://cdn.plot.ly/plotly-2.2.0.min.js")
			const PLOT = this ?? document.createElement("div");
		

		
			Plotly.react(PLOT, $(HypertextLiteral.JavaScript(PlotlyBase.json(plt))));


		
			const pluto_output = currentScript.parentElement.closest('pluto-output')

			const resizeObserver = new ResizeObserver(entries => {
				Plotly.Plots.resize(PLOT)
			})

			resizeObserver.observe(pluto_output)

			invalidation.then(() => {
				resizeObserver.disconnect()
			})
		
			return PLOT
			</script>
			</div>
	"""))
	end
end

# ╔═╡ ced63ea2-60c8-4913-a5d4-9e1b5ff47223
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 6571612f-00ad-4179-bb93-a48cc9804352
md"""
## Exercise 1. The Monetary Policy curve
"""

# ╔═╡ dac7fb80-10f7-4806-af0c-088637b79b69
md"""

*From the textbook.*

Assume the monetary policy curve is given by:  $\quad r = 1.5 + 0.75 \pi$.

a) Calculate the real interest rate when the inflation rate is at 0%, 2%, and 4%.

b) The following figure plots the monetary policy curve above. What do the three dotted points tell us?
"""

# ╔═╡ f1a259ac-e3c3-457a-9111-5eab394f452f
md"""
---
"""

# ╔═╡ 0ba1252d-d891-4835-b81a-d63210f5c87b
md"""
Firstly, let us insert the values of $\bar{r}$ and $\lambda$ into the notebook (1 is used as an index to refer the exercise 1):
"""

# ╔═╡ 3ced63a3-0e7a-4a7a-ae26-5e4edae88cac
r̄1 = 1.5 ;	λ1 = 0.75 ;

# ╔═╡ 6ea4d683-6825-4baa-bfe5-d8784f46420e
md"""
**(a)** These calculations can be done with a calculator, but a simple function will do it in a much easier way:
"""

# ╔═╡ f1acdec8-d55b-4b12-9b67-84c47be9378e
begin
	mp(π) = r̄1 .+ λ1 .* π
	π = [0 , 2 , 4]
	mp(π)
end

# ╔═╡ ce89db42-eabe-4d30-8d0a-4b06ad04145a
md"""
!!! answer "Answer (a)"

	For an inflation rate at 0%, 2%, and 4%, the real interest rate will be equal to 1.5%, 3%, and 4.5%, respectively.

"""

# ╔═╡ d38d5d7e-ef45-40e7-90c0-441fa4d30a24
begin
	r1 = 0.0:0.01:6.0
	
	π1 = - (r̄1 ./ λ1) .+  r1 ./ λ1 
	
	trace5_0 = scatter(; x = r1, y = π1, mode="lines" , line_color = "Brown", line_width = "3",
					name  = "MP")
	
	trace5_1 = scatter(; x = [1.5, 3.0, 4.5], y = [0.0, 2.0, 4.0], text =["1", "2" , "3"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Brown", textfont = attr(family="sans serif", size=16, color="black"))
	
	layout5_1 = Layout(;
					title_text ="MP curve", 
					title_x = 0.5,
					hovermode ="x",
                    xaxis=attr(title=" Real interest rate (r)", showgrid=true, zeroline=false),
                    xaxis_range = [0.0, 6.0],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false))

   p5_1 = Plot([trace5_0, trace5_1], layout5_1)
end

# ╔═╡ 0120645b-0386-4fdd-b7fd-ace3f58c2d4d
md"""
!!! answer "Answer (b)"

	They tell us the central bank's reaction to the three different inflation rates. If inflation increases, the central bank will increase real interest rates. By how much? It depends on how much the central bank dislikes inflation. The more the central bank dislikes inflation, the higher will be $\lambda$, and the higher the slope of the MP curve.

"""

# ╔═╡ 9690a57f-245b-4c2b-8ddc-0c9ef6b33ee8
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 1b35a949-7c83-4d64-aa69-90b42c5dfbfb
md"""
## Exercise 2. A shift in the Monetary Policy curve
"""

# ╔═╡ 5235a517-8947-4d57-a436-c48eb3de87c6
md"""
Refer to the monetary policy curve described in Problem 1. Assume now that the monetary policy curve is given by $r = 2.5 + 0.75 \pi$.

a) Using the slider below, plot the new monetary policy curve on the graph created in Problem 1.

b) Does the new monetary policy curve represent an autonomous tightening or loosening of monetary policy?

"""

# ╔═╡ 4c2622b9-191f-4e38-b5f4-c0eece8cb7af
md"""
!!! answer "Answer (a)"

	Setting the slider to $\bar{r}2=2.5$, the MP curve will shift to the right (see figure below).
"""

# ╔═╡ 6bf1aae8-5d3a-4aa2-b01c-d633602f3c55
md"""
`r̄2 = ` $(@bind r̄2 Slider(0.0:0.5:3.0, default=1.5, show_value=true))     
"""

# ╔═╡ 0fd242c3-a99f-41b3-a5ed-1c92f6d3b600
begin	
	π2_Δr̄2 = - (r̄2 ./  λ1) .+  r1 ./ λ1 
			
	trace5_2 = scatter(; x = r1, y = π2_Δr̄2, mode="lines" , line_color = "Purple", line_width = "3",
					 name  = "MP2")
	
	trace5_3 = scatter(; x = [1.5, 3.0, 4.5], y = [-4/3, 2/3, 8/3], text =["1''", "2''" , "3'"], 
					 textposition = "top center", name ="Eq.", mode="markers+text", 
					 marker_size= "12", marker_color="Purple", 
					 textfont = attr(family="sans serif", size=16, color="black"))
	
	trace5_4 = scatter(; x = [2.5, 4.0, 5.5], y = [0.0, 2.0, 4.0], text =["1'", "2'" , "3'"], 
					 textposition = "top center", name ="Eq.", mode="markers+text", 
					 marker_size= "12", marker_color="Purple", 
					 textfont = attr(family="sans serif", size=16, color="black"))
	
	layout5_4 = Layout(;
					title_text ="MP curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" Real interest rate (r)", showgrid=true, zeroline=false),
                    xaxis_range = [0.0, 6.0],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false))

	p5_4 = Plot([trace5_2, trace5_4, trace5_0, trace5_1], layout5_4)
end

# ╔═╡ 559be550-e0b0-4cd4-bd16-f7757bb190cb
md"""
!!! answer "Answer (b)"

	For the same level of inflation, the real interest rate is higher than before, which means that the central bank is tightening its monetary policy.
"""

# ╔═╡ 39562a1b-7a97-4ccd-b57c-9fb4bb15ba65
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 85981e5d-1519-4dca-8739-4023e58a0176
md"""
## Exercise 3. A comprehensive exercise
"""

# ╔═╡ b9a1a968-24f3-4a92-8341-614bf5a7e0ca
md"""
The following information is known about the aggregate demand and the central bank's behavior:
- Autonomous aggregate demand: $\overline{A} = 7.6$
- The multiplier: $m= 2.0$
- The set of parameters that gives the impact of $r$ upon aggregate demand is: $\phi=0.2$
- The MP curve is given by: 

$$r=\overline{r}+\lambda \pi$$
with $\overline{r}=2\%$ and $\lambda = 0.5$. Take into account that the value $2\%$ is expressed in percentage points, so that $2\%$ is just $2$. 

(a) Obtain the expression of the AD curve.

(b) The MP and the AD curves are represented graphically below. If inflation is $2\%$, what is the level of aggregate demand and the real interest rate set by the central bank?

c) The second figure below presents two points: 1 and 2. How do you explain the movement from points 1 to 2?

(d) What is the value of the nominal interest rate at both points? Is the Taylor principle satisfied in this exercise?

(e) Consider the central bank is losing control of the inflationary pressures in the economy. It takes a drastic measure by increasing $\overline{r}%$ to $3\%$. What happens to the MP and AD curves? Using the slider below, represent graphically.

"""

# ╔═╡ 22e8290e-c39f-41e2-94e5-d9c3b1585d88
Ā3 = 7.6 ;	λ3 = 0.5 ;	m3 = 2.0 ;	ϕ3 = 0.2 ;	r̄3 = 2.0 ;	

# ╔═╡ 3556cde6-e9b0-4dab-a8ae-dbc24f524b22
md"""
!!! hint

	Recall the IS curve:

	$$\tag{IS} Y=m \cdot \overline{A}-m \cdot \phi \cdot r$$

	Recall the MP curve: 

	$$\tag{MP} r=\overline{r}+\lambda \pi$$

	The AD curve is obtained by inserting the MP curve into the IS curve.

"""

# ╔═╡ ebb43969-4b22-4e8e-8694-6b47a8726b86
md"""
!!! answer "Answer (a)"

	Inserting the values of parameters and $\overline{A}$ into the IS and using the MP curve $(r = 2 + 0.5 \pi)$, we get:
	
	$$Y= 2.0 \times 7.6 - 2.0 \times 0.2 \times (2.0+0.5 \times \pi)$$
	$$\tag{AD} Y= 2.0 \times 7.6 - 0.8 - 0.2 \times \pi$$

"""

# ╔═╡ 894235bf-9515-4c82-b297-23e523764364
md"""
**(b)** The graphical representation is given by the following figures:
"""

# ╔═╡ 280f3cc5-0a2b-4cd3-bcfb-cba536877019
md"""
!!! answer "Answer (b)"

	Notice that using Pluto allows us to get the answer without having to do any numerical calculations. Using the mouse on top of the curves, we can easily see that if $\pi = 2\%$, then $r=3\%$, and Y = \$14 trillion.

"""

# ╔═╡ 644a26a3-0c5e-485c-ad2c-67afcc085d2d
md"""
**(c)** The graphical representation is
"""

# ╔═╡ 5bbebe42-69cd-474a-ad1a-8599fed108be
md"""
!!! answer "Answer (c)"

	If inflation increases from $\pi = 2\%$ to $\pi = 4\%$, then the central bank increases the real interest rate from $r=3\%$ to $r=4\%$, and aggregate demand will decline from  Y= \$14  trillion to Y = \$13.6 trillion.

	"""

# ╔═╡ fd292c9b-ed3e-4f91-aea6-4142e429163a
md"""
!!! answer "Answer (d)"

	The value of the nominal interest rate in both situations is obtained by using the Fisher equation:

	$$i =\pi + r$$

	Therefore:

	- If $r=3\%$ and $\pi = 2\%$, then $i= 5\%$.

	- If $r=4\%$ and $\pi = 4\%$, then $i= 8\%$.

	The Taylor principle is satisfied in this exercise because

	$$\Delta i>\Delta \pi$$
	$$3\% >2\%.$$

"""

# ╔═╡ e28e76fd-14d5-4d32-8d98-f8aac4734c43


# ╔═╡ 7cab8ef9-0744-46c1-ab50-18c78ecbe931
md"""
**(e)** This question involves an increase of $\bar{r}$ from 2% to 3%. Therefore, we have to move the slider according to this increment.  
"""

# ╔═╡ 1fb6c731-ff80-41fd-b866-0db9ad4ec361
md"""
`Δr̄3 = ` $(@bind Δr̄3 Slider(-2.0:0.25:2.0, default=0.0, show_value=true))     
"""

# ╔═╡ 5fe68e9d-3ff2-495e-b844-649e273ce758
md"""
!!! answer "Answer (e)"

	 This question involves a **shift** of the MP curve from points 1 to 1', or from 2 to 2'. This type of movement is entirely different from a movement along this curve as we had in question (c). Now, for the same inflation level, we have a higher level of the real interest rate and a lower level of aggregate demand.

"""

# ╔═╡ 0f86d6fa-bdd4-4018-b3af-39e0bb1288f7
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 6537fb39-c597-41ca-8b65-2d2912b0234a
md"""
## Exercise 4. Increases in aggregate demand
"""

# ╔═╡ a1dc0363-8871-49d9-afc8-16a2660ae5c3
md"""
*Fom the textbook*

What would be the effect on the aggregate demand curve of a large increase in U.S. net exports, or in public expenditures? Would any of those increases affect the monetary policy curve? Explain why or why not.
"""

# ╔═╡ 4cc6e64e-4858-4f2a-b363-e7619c060bf8
md"""
!!! answer

	**1.** Any of those increases will increase aggregate demand by shifting the AD curve to the right. 

	**2.** If any of those increases lead to a **small rise** in inflation that can be accommodated with a small increase in the real interest rate, then none of them will shift the MP curve. 

	**3.** However, if any of those increases are so large that they will lead to a **large rise** in inflation, the central bank will have no other choice but promote a substantial increase in the real interest rate. In this case, there will be a shift of the MP curve to the right.

	"""

# ╔═╡ b99a4854-6671-4296-99b9-977af285812b
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ e2bd0872-89de-4532-9be1-4b39c241133b
md"""
## Exercise 5. Monetary policy and employment
"""

# ╔═╡ 51b3a17c-2454-4cff-a1cc-a2f27d047f91
md"""

*From the textbook.*


Suppose U.S. aggregate output is still below potential by 2018, when a new Fed chair is appointed. Suppose his or her approach to monetary policy can be summarized by the following statement: “I care only about increasing employment; inflation has been at very low levels for quite some time; my priority is to ease monetary policy to promote employment.”

(a) Would you expect the monetary policy curve to shift upward or downward?

(b) What would be the effect on the aggregate demand curve?
"""

# ╔═╡ 5ae9e073-7506-405e-a994-de7a08337bfa
md"""
	!!! answers

	    **(a)** The sentence shows very clearly that the economy is either in a recession or not in very good shape ("aggregate output is still below potential by 2018"), and inflation is low and stable, while an increase in employment is the crucial priority. So, everything points at a significant reduction in the real interest rate. Therefore, one should expect a shift of the MP curve to the left (or upwards). 

		**(b)** A shift of the MP curve upwards will lead to an increase in aggregate demand and will shift the AD curve upward as well. 
	"""

# ╔═╡ a71621ac-f4f6-4b54-8486-b7bdaa8d29ef
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 747b89f4-fd33-4fc6-bd24-83b8860a5a5e
md"""
## Exercise 6. The twin goals of monetary policy
"""

# ╔═╡ b293a9cc-6c2e-4e8d-9445-9d377b83f676
md"""
*From the textbook "Macroeconomics", Fourth Edition, by Charles I. Jones, 2018, Pearson.*

Your day as chair of the Fed (I): Suppose you are appointed to chair the Federal Reserve. Your twin goals are to maintain low inflation and to stabilize economic activity — that is, to keep GDP at potential. Why are these appropriate goals for monetary policy? (Hint: What happens if the economy booms? Or in a deep recession?)

"""

# ╔═╡ b8c26c04-3448-4e47-8fb6-9ba6f13ab210
md"""
!!! answer
	**1.** These are appropriate goals because they reflect an almost idealistic situation. If GDP is at its potential level, no inflationary or disinflationary pressures exist in the economy, so inflation is stable and low. On the other hand, it also means that unemployment is at its natural level, which means no economic policy measure can reduce unemployment.  

	**2.** If we want the economy to be above potential GDP, we have to accept inflationary pressures to reduce unemployment below its natural rate. This is bad because the central bank can lose control of inflation. 

	**3.** If an economy becomes trapped in a situation with GDP too much below its potential level for a significant time, it may end up in a deflationary situation. This situation is the worst of all evils in macroeconomics (as we will see better in Week 10).
"""

# ╔═╡ 4ae438b1-0752-411f-ae55-52479f518c57
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ d04e8abb-ae8e-4297-9734-533d9274274e
md"""
## Exercise 7. Large shocks and monetary policy
"""

# ╔═╡ 94b9b913-2a22-4960-992a-699f18954be2
md"""
*Adapted from the textbook "Macroeconomics", Fourth Edition, by Charles I. Jones, 2018, Pearson.*

Your day as chair of the Fed (II): With the goal of **stabilizing output**, explain how and why you would change the real interest rate in response to the following large shocks. Describe what happens in the short run using the AD-MP diagram (no need to use numerical simulations)

(a) Consumers become extremely pessimistic about the state of the economy and future productivity growth.

(b) Improvements in information technology increase productivity and therefore increase the marginal product of capital (MPK).

(c) A booming economy in Europe leads to an unexpected increase in the demand for U.S. goods by European consumers.

(d) A housing bubble bursts, so that housing prices fall by 20% and new home sales drop sharply.

"""

# ╔═╡ cbc7414b-8788-4527-a766-abbc49512eaa
md"""
!!! answers

	**(a)** Autonomous consumption will go down, and the AD curve will shift to the left. To stabilize GDP, the central bank will react by imposing a large reduction in real-interest rates, so the MP curve will move to the left until it forces the AD curve to return to its initial position.

	**(b)** An increase in the MPK will increase autonomous investment expenditures, and the AD curve will shift to the right. The central bank will have to increase the real interest rate, and the MP curve will change to the right until it forces the AD curve to return to its initial position. 

	**(c)** Net exports increase ... similar to the previous answer.

	**(d)** If a housing bubble bursts, investment expenditures drop significantly because the financial friction increases and autonomous investment decreases. Private consumption expenditures also go down (consumers' confidence is seriously affected). These two effects will produce a vast drop in aggregate demand: the AD curve drastically shifts to the left. The central bank will have to reduce real-interest rates so much that they may need to reach negative values to fight a deep recession. If the central bank lowers the real interest rate by a lot (so the MP curve shifts to the left by a lot), it may force the AD curve to return to its initial position. Usually, as we will see in Week 10, that reduction in interest rates is not enough: the central bank will have to resort to "unconventional" monetary policy measures.

"""

# ╔═╡ 7a82dcaf-b775-479a-94be-23a957ff0d9d
md"""
## Exercise 8. UK Mini-Budget & Budget Responsability
"""

# ╔═╡ 97a132ae-a371-4449-a4bd-e2ce469cffca
md"""
Illustration by Chris Riddell
"""

# ╔═╡ 48a35af0-ab1d-4e6b-a195-0d3d885b5652
Resource("https://vivaldomendes.org/images/depot/tax_cuts.webp", :width=>600)

# ╔═╡ 8b8bdbd0-6148-4a9d-b2e1-d6625cb84ac4
md"""
With an inflation rate of 9.9% (the highest in more than forty years), a long and brutal war in Europe, and energy markets in turmoil, the UK government promised “a new approach for a new era”. On 23 September 2022, the government announced the highest unfunded tax cuts and massive increases in public spending in more than half a century. The reaction of markets, international institutions, and commentators was dismayed.

"The IMF said fiscal stimulus is inappropriate given the inflation pressures in the UK economy, and the package risks making life harder for the Bank of England. Moody’s forecast that it will lower economic growth -- contradicting the view from Chancellor of the Exchequer Kwasi Kwarteng -- by pushing up interest rates. ", Bloomberg, 27 September 2022

Using the concepts of aggregate demand (AD curve) and the MP curve, why does the plan by the UK government look "on a course of sheer madness", as Ambrose Evans-Pritchard has put in The Telegraph, 29 September 2022?

"""

# ╔═╡ cef3ec74-5394-494c-af80-a29f469abe8e
md"""
!!! answer "Suggested answer"

	Before the fiscal stimulus package was announced, the Bank of England (BoE) had already aggressively raised interest rates. It was expected to continue to do so for the time coming because inflation was very high and displaying an upward trend. That package will put more inflationary pressures on the economy, forcing the central bank to raise rates much more than expected. 

	The UK government is forcing the AD curve to take a big jump to the right, increasing demand and GDP. However, because inflation was very high, the BoE had already aggressively raised interest rates, so the MP curve moved to the right to reduce aggregate demand and inflation. By this action, the BoE is trying to force the AD curve to shift to the left, and now they have to do it even more aggressively than initially expected. 

	There was also an extremely adverse reaction by the markets: the pound hit a record low against the dollar after the sweeping tax cuts were announced, and the yields on the UK public debt suffered a dramatic increase forcing the BoE to intervene. Both reactions will force the BoE to raise interest rates much higher than would otherwise be required. Instead of GDP growth, a recession should be the outcome.

"""

# ╔═╡ d4b320e8-b03a-4f44-b842-f7f6bc4724b9
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ 2c7c6904-52cd-459b-b459-daf6bdb4f91d
md"""
## Exercise 9. A very simple problem
"""

# ╔═╡ 40979b66-ab86-45ac-8102-64704d8bd613
md"""

*From the textbook.*

Suppose the monetary policy curve is given by $r = 1.5 + 0.75 \pi$, and the IS curve is given by $Y = 13 - r$.

(a) Find the expression for the aggregate demand curve.

(b) Calculate aggregate output when the inflation rate is at 2%, 3%, and 4%.

"""

# ╔═╡ 602a7f0f-6b1b-445c-8d3c-0887f858b1e0
md"""
!!! answers

    **(a)** From a numerical point of view, this is a very easy problem to solve. AD:  $Y = 11.5 - 0.75 \pi$.

	**(b)** This is extremely easy, so ... (if you want to use Julia, see Exercise 1)

	"""

# ╔═╡ 81c00071-63c2-48ff-96a9-b3ebcf8cc44f
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ fd4a0888-c180-4f68-98e2-40061bc27354
md"""
## Auxiliary cells (do not delete)
"""

# ╔═╡ e0a35f3d-82d5-4315-ae83-27329529cf15
html"""<style>
main {
    max-width: 900px;
    align-self: flex-start;
    margin-left: 100px;
}
"""

# ╔═╡ 18195c33-8d9e-431f-936b-b439a8b840a8
TableOfContents()

# ╔═╡ 31068d0a-0945-49a3-8c72-7a5ebb288db2
md"""
##### Exercise 3: supporting cells
"""

# ╔═╡ 1d1f906f-a672-4a0f-a35c-db1dfb0763fc
begin

	function fig_3b1()
		
		r3 = 1.5:0.01:7.0
	
		π3 = - (r̄3 ./ λ3) .+  r3 ./ λ3
	
		trace5_5 = scatter(; x = r3, y = π3, mode="lines" , line_color = "Brown", line_width = "3",
					name  = "MP")
	
		layout5_5 = Layout(;
					title_text ="MP curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" Real interest rate (r)", showgrid=true, zeroline=false),
                    xaxis_range = [1.5, 6.5],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-1 ,9])

   		p5_5 = Plot([trace5_5], layout5_5)
		
	end
	#fig_3b1()
end

# ╔═╡ 89be032b-c88d-4de8-9f66-4acbdd3dec3d
begin
	function fig_3b2()
		
		Y3 = 12.0:0.01:14.6
		πdmax3 = (Ā3 / (λ3 * ϕ3)) - r̄3/λ3 		# The AD curve y-axis intercept 
	
		πd3 = πdmax3 .- ((1 ./(m3 .* ϕ3 .* λ3)) .* Y3)
	
		trace5_6 = scatter(; x = Y3, y = πd3, mode="lines" , line_color = "Blue", line_width = "3",
					name  = "AD")
	
	
		layout5_6 = Layout(;
					title_text ="AD curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" GDP (Y)", showgrid=true, zeroline=false),
                    xaxis_range = [12.8, 14.6],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-1 ,9])

   		p5_6 = Plot([trace5_6], layout5_6)
	end
	#fig_3b2()
		
end

# ╔═╡ 4e65d2d8-c706-4a65-92ad-477e17ad39cb
begin
	p5_6A = [fig_3b1() fig_3b2()]
	relayout!(p5_6A, hovermode = "y")
	p5_6A
end

# ╔═╡ 0d3c51a2-892b-4db1-a2af-fcddb75ef635
begin
	
	function fig_3c1()
		
		r3 = 1.5:0.01:7.0
		π3 = - (r̄3 ./ λ3) .+  r3 ./ λ3
	
		trace5_7 = scatter(; x = r3, y = π3, mode="lines" , line_color = "Brown", line_width = "3",
					name  = "MP")
	
		trace5_8 = scatter(; x = [3.0 , 4.0], y = [2.0 , 4.0], text =["1", "2"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Brown", textfont = attr(family="sans serif", size=16, color="black"))
	
	
		layout5_8 = Layout(;
					title_text ="MP curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" Real interest rate (r)", showgrid=true, zeroline=false),
                    xaxis_range = [1.5, 6.5],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-1 ,9])

   		p5_8 = Plot([trace5_7, trace5_8], layout5_8)
	end
	#fig_3c1()
end

# ╔═╡ 1a055e0b-d23e-4d85-89bf-5813127e88ec
begin

	function fig_3c2()
		
		Y3 = 12.0:0.01:14.6

		πdmax3 = (Ā3 / (λ3 * ϕ3)) - r̄3/λ3 		# The AD curve y-axis intercept 

		πd3 = πdmax3 .- ((1 ./(m3 .* ϕ3 .* λ3)) .* Y3)
	
		trace5_9 = scatter(; x = Y3, y = πd3, mode="lines" , line_color = "Blue", line_width = "3",
					name  = "AD")
	
		trace5_10 = scatter(; x = [14.0 , 13.6], y = [2.0 , 4.0], text =["1", "2"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Blue", textfont = attr(family="sans serif", size=16, color="black"))
	
	
		layout5_10 = Layout(;
					title_text ="AD curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" GDP (Y)", showgrid=true, zeroline=false),
                    xaxis_range = [12.8, 14.6],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-1 ,9])

   		p5_10 = Plot([trace5_9, trace5_10], layout5_10)
	end

	#fig_3c2()
		
end

# ╔═╡ bb50e179-3e60-4fab-9be4-3b56f254c710
begin
	p5_10A = [fig_3c1() fig_3c2()]
	relayout!(p5_10A, hovermode = "x")
	p5_10A
end

# ╔═╡ 4a4d51cb-8626-4d97-91aa-c4ce37b0a210
begin
	
	function fig_3e1()
		
		r3 = 1.5:0.01:7.0
		π3 = - (r̄3 ./ λ3) .+  r3 ./ λ3
		r3_Δr̄3 = 1.5:0.01:7.0
		π3_Δr̄3 = - ((r̄3 .+ Δr̄3) ./ λ3) .+  r3 ./ λ3

		trace5_7 = scatter(; x = r3, y = π3, mode="lines" , line_color = "Brown", line_width = "3",
					name  = "MP")
		
		trace5_8 = scatter(; x = [3.0 , 4.0], y = [2.0 , 4.0], text =["1", "2"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Brown", textfont = attr(family="sans serif", size=16, color="black"))
	
		trace5_11 = scatter(; x = r3_Δr̄3, y = π3_Δr̄3, mode="lines" , line_color = "Purple", 
					line_width = "3", name  = "MP1")
	
		trace5_11A = scatter(; x = [4.0 , 5.0], y = [2.0 , 4.0], text =["1'", "2'"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Purple", textfont = attr(family="sans serif", size=16, color="black"))
	
		layout5_11 = Layout(;
					title_text ="MP curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" Real interest rate (r)", showgrid=true, zeroline=false),
                    xaxis_range = [1.5, 6.5],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-1 ,9])

   		p5_11 = Plot([trace5_11, trace5_7, trace5_11A, trace5_8], layout5_11)
		
	end
	
	#fig_3e1()
end

# ╔═╡ f091ac64-be2c-4cc8-a7d3-34abde9fd5d4
begin
	function fig_3e2()
		
		Y3_Δr̄3 = 12.0:0.01:14.6

		πdmax3 = (Ā3 / (λ3 * ϕ3)) - r̄3/λ3 		# The AD curve y-axis intercept
	
		Y3 = 12.0:0.01:14.6

		πd3 = πdmax3 .- ((1 ./(m3 .* ϕ3 .* λ3)) .* Y3)

		πdmax3_Δr̄3 = (Ā3 / (λ3 * ϕ3)) - (r̄3 + Δr̄3) / λ3
		
		πd3_Δr̄3 = πdmax3_Δr̄3 .- ((1 ./(m3 .* ϕ3 .* λ3)) .* Y3_Δr̄3)

		trace5_9 = scatter(; x = Y3, y = πd3, mode="lines" , line_color = "Blue", line_width = "3",
					name  = "AD")

		trace5_10 = scatter(; x = [14.0 , 13.6], y = [2.0 , 4.0], text =["1", "2"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Blue", textfont = attr(family="sans serif", size=16, color="black"))
	
		trace5_12 = scatter(; x = Y3_Δr̄3 , y = πd3_Δr̄3, mode="lines" , line_color = "Violet", 
					line_width = "3", name  = "AD2")
	
		trace5_12A = scatter(; x = [13.6 , 13.2], y = [2.0 , 4.0], text =["1'", "2'"], 
					textposition = "top center", name ="Eq.", mode="markers+text", marker_size= "12",
					marker_color="Violet", textfont = attr(family="sans serif", size=16, color="black"))
		
		layout5_12 = Layout(;
					title_text ="AD curve", 
					title_x = 0.5,
					hovermode = "x",
                    xaxis=attr(title=" GDP (Y)", showgrid=true, zeroline=false),
                    xaxis_range = [12.8, 14.6],
                    yaxis=attr(title="Rate of inflation (π)", zeroline = false),
					yaxis_range=[-1 , 9])

   		p5_12 = Plot([trace5_12, trace5_9, trace5_12A, trace5_10], layout5_12)

	end

	#fig_3e2()
	
end

# ╔═╡ 51818620-e43c-4db4-813b-ed5c4562b62c
begin
	p5_12A = [fig_3e1() fig_3e2()]
	relayout!(p5_12A, hovermode = "x")
	p5_12A
end


# ╔═╡ 00000000-0000-0000-0000-000000000001
PLUTO_PROJECT_TOML_CONTENTS = """
[deps]
CSV = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
Dates = "ade2ca70-3891-5945-98fb-dc099432e06a"
HypertextLiteral = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
PeriodicalDates = "276e7ca9-e0d7-440b-97bc-a6ae82f545b1"
PlotlyBase = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
PlutoUI = "7f904dfe-b85e-4ff6-b463-dae2292396a8"

[compat]
CSV = "~0.10.4"
DataFrames = "~1.3.6"
HypertextLiteral = "~0.9.4"
PeriodicalDates = "~2.0.0"
PlotlyBase = "~0.8.19"
PlutoUI = "~0.7.43"
"""

# ╔═╡ 00000000-0000-0000-0000-000000000002
PLUTO_MANIFEST_TOML_CONTENTS = """
# This file is machine-generated - editing it directly is not advised

julia_version = "1.8.1"
manifest_format = "2.0"
project_hash = "d6cd2873a2e9b3a5940b54ea09d8f1ec5d85bf86"

[[deps.AbstractPlutoDingetjes]]
deps = ["Pkg"]
git-tree-sha1 = "8eaf9f1b4921132a4cff3f36a1d9ba923b14a481"
uuid = "6e696c72-6542-2067-7265-42206c756150"
version = "1.1.4"

[[deps.ArgTools]]
uuid = "0dad84c5-d112-42e6-8d28-ef12dabb789f"
version = "1.1.1"

[[deps.Artifacts]]
uuid = "56f22d72-fd6d-98f1-02f0-08ddc0907c33"

[[deps.Base64]]
uuid = "2a0f44e3-6c83-55bd-87e4-b1978d98bd5f"

[[deps.CSV]]
deps = ["CodecZlib", "Dates", "FilePathsBase", "InlineStrings", "Mmap", "Parsers", "PooledArrays", "SentinelArrays", "Tables", "Unicode", "WeakRefStrings"]
git-tree-sha1 = "873fb188a4b9d76549b81465b1f75c82aaf59238"
uuid = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
version = "0.10.4"

[[deps.ChainRulesCore]]
deps = ["Compat", "LinearAlgebra", "SparseArrays"]
git-tree-sha1 = "e7ff6cadf743c098e08fca25c91103ee4303c9bb"
uuid = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
version = "1.15.6"

[[deps.ChangesOfVariables]]
deps = ["ChainRulesCore", "LinearAlgebra", "Test"]
git-tree-sha1 = "38f7a08f19d8810338d4f5085211c7dfa5d5bdd8"
uuid = "9e997f8a-9a97-42d5-a9f1-ce6bfc15e2c0"
version = "0.1.4"

[[deps.CodecZlib]]
deps = ["TranscodingStreams", "Zlib_jll"]
git-tree-sha1 = "ded953804d019afa9a3f98981d99b33e3db7b6da"
uuid = "944b1d66-785c-5afd-91f1-9de20f533193"
version = "0.7.0"

[[deps.ColorSchemes]]
deps = ["ColorTypes", "ColorVectorSpace", "Colors", "FixedPointNumbers", "Random"]
git-tree-sha1 = "1fd869cc3875b57347f7027521f561cf46d1fcd8"
uuid = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
version = "3.19.0"

[[deps.ColorTypes]]
deps = ["FixedPointNumbers", "Random"]
git-tree-sha1 = "eb7f0f8307f71fac7c606984ea5fb2817275d6e4"
uuid = "3da002f7-5984-5a60-b8a6-cbb66c0b333f"
version = "0.11.4"

[[deps.ColorVectorSpace]]
deps = ["ColorTypes", "FixedPointNumbers", "LinearAlgebra", "SpecialFunctions", "Statistics", "TensorCore"]
git-tree-sha1 = "d08c20eef1f2cbc6e60fd3612ac4340b89fea322"
uuid = "c3611d14-8923-5661-9e6a-0046d554d3a4"
version = "0.9.9"

[[deps.Colors]]
deps = ["ColorTypes", "FixedPointNumbers", "Reexport"]
git-tree-sha1 = "417b0ed7b8b838aa6ca0a87aadf1bb9eb111ce40"
uuid = "5ae59095-9a9b-59fe-a467-6f913c188581"
version = "0.12.8"

[[deps.Compat]]
deps = ["Dates", "LinearAlgebra", "UUIDs"]
git-tree-sha1 = "5856d3031cdb1f3b2b6340dfdc66b6d9a149a374"
uuid = "34da2185-b29b-5c13-b0c7-acf172513d20"
version = "4.2.0"

[[deps.CompilerSupportLibraries_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "e66e0078-7015-5450-92f7-15fbd957f2ae"
version = "0.5.2+0"

[[deps.Crayons]]
git-tree-sha1 = "249fe38abf76d48563e2f4556bebd215aa317e15"
uuid = "a8cc5b0e-0ffa-5ad4-8c14-923d3ee1735f"
version = "4.1.1"

[[deps.DataAPI]]
git-tree-sha1 = "1106fa7e1256b402a86a8e7b15c00c85036fef49"
uuid = "9a962f9c-6df0-11e9-0e5d-c546b8b5ee8a"
version = "1.11.0"

[[deps.DataFrames]]
deps = ["Compat", "DataAPI", "Future", "InvertedIndices", "IteratorInterfaceExtensions", "LinearAlgebra", "Markdown", "Missings", "PooledArrays", "PrettyTables", "Printf", "REPL", "Reexport", "SortingAlgorithms", "Statistics", "TableTraits", "Tables", "Unicode"]
git-tree-sha1 = "db2a9cb664fcea7836da4b414c3278d71dd602d2"
uuid = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
version = "1.3.6"

[[deps.DataStructures]]
deps = ["Compat", "InteractiveUtils", "OrderedCollections"]
git-tree-sha1 = "d1fff3a548102f48987a52a2e0d114fa97d730f0"
uuid = "864edb3b-99cc-5e75-8d2d-829cb0a9cfe8"
version = "0.18.13"

[[deps.DataValueInterfaces]]
git-tree-sha1 = "bfc1187b79289637fa0ef6d4436ebdfe6905cbd6"
uuid = "e2d170a0-9d28-54be-80f0-106bbe20a464"
version = "1.0.0"

[[deps.Dates]]
deps = ["Printf"]
uuid = "ade2ca70-3891-5945-98fb-dc099432e06a"

[[deps.DelimitedFiles]]
deps = ["Mmap"]
uuid = "8bb1440f-4735-579b-a4ab-409b98df4dab"

[[deps.DocStringExtensions]]
deps = ["LibGit2"]
git-tree-sha1 = "5158c2b41018c5f7eb1470d558127ac274eca0c9"
uuid = "ffbed154-4ef7-542d-bbb7-c09d3a79fcae"
version = "0.9.1"

[[deps.Downloads]]
deps = ["ArgTools", "FileWatching", "LibCURL", "NetworkOptions"]
uuid = "f43a241f-c20a-4ad4-852c-f6b1247861c6"
version = "1.6.0"

[[deps.FilePathsBase]]
deps = ["Compat", "Dates", "Mmap", "Printf", "Test", "UUIDs"]
git-tree-sha1 = "e27c4ebe80e8699540f2d6c805cc12203b614f12"
uuid = "48062228-2e41-5def-b9a4-89aafe57970f"
version = "0.9.20"

[[deps.FileWatching]]
uuid = "7b1f6079-737a-58dc-b8bc-7a2ca5c1b5ee"

[[deps.FixedPointNumbers]]
deps = ["Statistics"]
git-tree-sha1 = "335bfdceacc84c5cdf16aadc768aa5ddfc5383cc"
uuid = "53c48c17-4a7d-5ca2-90c5-79b7896eea93"
version = "0.8.4"

[[deps.Formatting]]
deps = ["Printf"]
git-tree-sha1 = "8339d61043228fdd3eb658d86c926cb282ae72a8"
uuid = "59287772-0a20-5a39-b81b-1366585eb4c0"
version = "0.4.2"

[[deps.Future]]
deps = ["Random"]
uuid = "9fa8497b-333b-5362-9e8d-4d0656e87820"

[[deps.Hyperscript]]
deps = ["Test"]
git-tree-sha1 = "8d511d5b81240fc8e6802386302675bdf47737b9"
uuid = "47d2ed2b-36de-50cf-bf87-49c2cf4b8b91"
version = "0.0.4"

[[deps.HypertextLiteral]]
deps = ["Tricks"]
git-tree-sha1 = "c47c5fa4c5308f27ccaac35504858d8914e102f9"
uuid = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
version = "0.9.4"

[[deps.IOCapture]]
deps = ["Logging", "Random"]
git-tree-sha1 = "f7be53659ab06ddc986428d3a9dcc95f6fa6705a"
uuid = "b5f81e59-6552-4d32-b1f0-c071b021bf89"
version = "0.2.2"

[[deps.InlineStrings]]
deps = ["Parsers"]
git-tree-sha1 = "d19f9edd8c34760dca2de2b503f969d8700ed288"
uuid = "842dd82b-1e85-43dc-bf29-5d0ee9dffc48"
version = "1.1.4"

[[deps.InteractiveUtils]]
deps = ["Markdown"]
uuid = "b77e0a4c-d291-57a0-90e8-8db25a27a240"

[[deps.InverseFunctions]]
deps = ["Test"]
git-tree-sha1 = "b3364212fb5d870f724876ffcd34dd8ec6d98918"
uuid = "3587e190-3f89-42d0-90ee-14403ec27112"
version = "0.1.7"

[[deps.InvertedIndices]]
git-tree-sha1 = "bee5f1ef5bf65df56bdd2e40447590b272a5471f"
uuid = "41ab1584-1d38-5bbf-9106-f11c6c58b48f"
version = "1.1.0"

[[deps.IrrationalConstants]]
git-tree-sha1 = "7fd44fd4ff43fc60815f8e764c0f352b83c49151"
uuid = "92d709cd-6900-40b7-9082-c6be49f344b6"
version = "0.1.1"

[[deps.IteratorInterfaceExtensions]]
git-tree-sha1 = "a3f24677c21f5bbe9d2a714f95dcd58337fb2856"
uuid = "82899510-4779-5014-852e-03e436cf321d"
version = "1.0.0"

[[deps.JLLWrappers]]
deps = ["Preferences"]
git-tree-sha1 = "abc9885a7ca2052a736a600f7fa66209f96506e1"
uuid = "692b3bcd-3c85-4b1f-b108-f13ce0eb3210"
version = "1.4.1"

[[deps.JSON]]
deps = ["Dates", "Mmap", "Parsers", "Unicode"]
git-tree-sha1 = "3c837543ddb02250ef42f4738347454f95079d4e"
uuid = "682c06a0-de6a-54ab-a142-c8b1cf79cde6"
version = "0.21.3"

[[deps.LaTeXStrings]]
git-tree-sha1 = "f2355693d6778a178ade15952b7ac47a4ff97996"
uuid = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
version = "1.3.0"

[[deps.LibCURL]]
deps = ["LibCURL_jll", "MozillaCACerts_jll"]
uuid = "b27032c2-a3e7-50c8-80cd-2d36dbcbfd21"
version = "0.6.3"

[[deps.LibCURL_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll", "Zlib_jll", "nghttp2_jll"]
uuid = "deac9b47-8bc7-5906-a0fe-35ac56dc84c0"
version = "7.84.0+0"

[[deps.LibGit2]]
deps = ["Base64", "NetworkOptions", "Printf", "SHA"]
uuid = "76f85450-5226-5b5a-8eaa-529ad045b433"

[[deps.LibSSH2_jll]]
deps = ["Artifacts", "Libdl", "MbedTLS_jll"]
uuid = "29816b5a-b9ab-546f-933c-edad1886dfa8"
version = "1.10.2+0"

[[deps.Libdl]]
uuid = "8f399da3-3557-5675-b5ff-fb832c97cbdb"

[[deps.LinearAlgebra]]
deps = ["Libdl", "libblastrampoline_jll"]
uuid = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"

[[deps.LogExpFunctions]]
deps = ["ChainRulesCore", "ChangesOfVariables", "DocStringExtensions", "InverseFunctions", "IrrationalConstants", "LinearAlgebra"]
git-tree-sha1 = "94d9c52ca447e23eac0c0f074effbcd38830deb5"
uuid = "2ab3a3ac-af41-5b50-aa03-7779005ae688"
version = "0.3.18"

[[deps.Logging]]
uuid = "56ddb016-857b-54e1-b83d-db4d58db5568"

[[deps.Markdown]]
deps = ["Base64"]
uuid = "d6f4376e-aef5-505a-96c1-9c027394607a"

[[deps.MbedTLS_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "c8ffd9c3-330d-5841-b78e-0817d7145fa1"
version = "2.28.0+0"

[[deps.Missings]]
deps = ["DataAPI"]
git-tree-sha1 = "bf210ce90b6c9eed32d25dbcae1ebc565df2687f"
uuid = "e1d29d7a-bbdc-5cf2-9ac0-f12de2c33e28"
version = "1.0.2"

[[deps.Mmap]]
uuid = "a63ad114-7e13-5084-954f-fe012c677804"

[[deps.MozillaCACerts_jll]]
uuid = "14a3606d-f60d-562e-9121-12d972cd8159"
version = "2022.2.1"

[[deps.NetworkOptions]]
uuid = "ca575930-c2e3-43a9-ace4-1e988b2c1908"
version = "1.2.0"

[[deps.OpenBLAS_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "Libdl"]
uuid = "4536629a-c528-5b80-bd46-f80d51c5b363"
version = "0.3.20+0"

[[deps.OpenLibm_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "05823500-19ac-5b8b-9628-191a04bc5112"
version = "0.8.1+0"

[[deps.OpenSpecFun_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "JLLWrappers", "Libdl", "Pkg"]
git-tree-sha1 = "13652491f6856acfd2db29360e1bbcd4565d04f1"
uuid = "efe28fd5-8261-553b-a9e1-b2916fc3738e"
version = "0.5.5+0"

[[deps.OrderedCollections]]
git-tree-sha1 = "85f8e6578bf1f9ee0d11e7bb1b1456435479d47c"
uuid = "bac558e1-5e72-5ebc-8fee-abe8a469f55d"
version = "1.4.1"

[[deps.Parameters]]
deps = ["OrderedCollections", "UnPack"]
git-tree-sha1 = "34c0e9ad262e5f7fc75b10a9952ca7692cfc5fbe"
uuid = "d96e819e-fc66-5662-9728-84c9c7592b0a"
version = "0.12.3"

[[deps.Parsers]]
deps = ["Dates"]
git-tree-sha1 = "3d5bf43e3e8b412656404ed9466f1dcbf7c50269"
uuid = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"
version = "2.4.0"

[[deps.PeriodicalDates]]
deps = ["Dates", "Printf", "RecipesBase"]
git-tree-sha1 = "e616941f8093e50a373e7d51875143213f82eca4"
uuid = "276e7ca9-e0d7-440b-97bc-a6ae82f545b1"
version = "2.0.0"

[[deps.Pkg]]
deps = ["Artifacts", "Dates", "Downloads", "LibGit2", "Libdl", "Logging", "Markdown", "Printf", "REPL", "Random", "SHA", "Serialization", "TOML", "Tar", "UUIDs", "p7zip_jll"]
uuid = "44cfe95a-1eb2-52ea-b672-e2afdf69b78f"
version = "1.8.0"

[[deps.PlotlyBase]]
deps = ["ColorSchemes", "Dates", "DelimitedFiles", "DocStringExtensions", "JSON", "LaTeXStrings", "Logging", "Parameters", "Pkg", "REPL", "Requires", "Statistics", "UUIDs"]
git-tree-sha1 = "56baf69781fc5e61607c3e46227ab17f7040ffa2"
uuid = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
version = "0.8.19"

[[deps.PlutoUI]]
deps = ["AbstractPlutoDingetjes", "Base64", "ColorTypes", "Dates", "Hyperscript", "HypertextLiteral", "IOCapture", "InteractiveUtils", "JSON", "Logging", "Markdown", "Random", "Reexport", "UUIDs"]
git-tree-sha1 = "2777a5c2c91b3145f5aa75b61bb4c2eb38797136"
uuid = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
version = "0.7.43"

[[deps.PooledArrays]]
deps = ["DataAPI", "Future"]
git-tree-sha1 = "a6062fe4063cdafe78f4a0a81cfffb89721b30e7"
uuid = "2dfb63ee-cc39-5dd5-95bd-886bf059d720"
version = "1.4.2"

[[deps.Preferences]]
deps = ["TOML"]
git-tree-sha1 = "47e5f437cc0e7ef2ce8406ce1e7e24d44915f88d"
uuid = "21216c6a-2e73-6563-6e65-726566657250"
version = "1.3.0"

[[deps.PrettyTables]]
deps = ["Crayons", "Formatting", "Markdown", "Reexport", "Tables"]
git-tree-sha1 = "dfb54c4e414caa595a1f2ed759b160f5a3ddcba5"
uuid = "08abe8d2-0d0c-5749-adfa-8a2ac140af0d"
version = "1.3.1"

[[deps.Printf]]
deps = ["Unicode"]
uuid = "de0858da-6303-5e67-8744-51eddeeeb8d7"

[[deps.REPL]]
deps = ["InteractiveUtils", "Markdown", "Sockets", "Unicode"]
uuid = "3fa0cd96-eef1-5676-8a61-b3b8758bbffb"

[[deps.Random]]
deps = ["SHA", "Serialization"]
uuid = "9a3f8284-a2c9-5f02-9a11-845980a1fd5c"

[[deps.RecipesBase]]
deps = ["SnoopPrecompile"]
git-tree-sha1 = "612a4d76ad98e9722c8ba387614539155a59e30c"
uuid = "3cdcf5f2-1ef4-517c-9805-6587b60abb01"
version = "1.3.0"

[[deps.Reexport]]
git-tree-sha1 = "45e428421666073eab6f2da5c9d310d99bb12f9b"
uuid = "189a3867-3050-52da-a836-e630ba90ab69"
version = "1.2.2"

[[deps.Requires]]
deps = ["UUIDs"]
git-tree-sha1 = "838a3a4188e2ded87a4f9f184b4b0d78a1e91cb7"
uuid = "ae029012-a4dd-5104-9daa-d747884805df"
version = "1.3.0"

[[deps.SHA]]
uuid = "ea8e919c-243c-51af-8825-aaa63cd721ce"
version = "0.7.0"

[[deps.SentinelArrays]]
deps = ["Dates", "Random"]
git-tree-sha1 = "c0f56940fc967f3d5efed58ba829747af5f8b586"
uuid = "91c51154-3ec4-41a3-a24f-3f23e20d615c"
version = "1.3.15"

[[deps.Serialization]]
uuid = "9e88b42a-f829-5b0c-bbe9-9e923198166b"

[[deps.SnoopPrecompile]]
git-tree-sha1 = "f604441450a3c0569830946e5b33b78c928e1a85"
uuid = "66db9d55-30c0-4569-8b51-7e840670fc0c"
version = "1.0.1"

[[deps.Sockets]]
uuid = "6462fe0b-24de-5631-8697-dd941f90decc"

[[deps.SortingAlgorithms]]
deps = ["DataStructures"]
git-tree-sha1 = "b3363d7460f7d098ca0912c69b082f75625d7508"
uuid = "a2af1166-a08f-5f64-846c-94a0d3cef48c"
version = "1.0.1"

[[deps.SparseArrays]]
deps = ["LinearAlgebra", "Random"]
uuid = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"

[[deps.SpecialFunctions]]
deps = ["ChainRulesCore", "IrrationalConstants", "LogExpFunctions", "OpenLibm_jll", "OpenSpecFun_jll"]
git-tree-sha1 = "d75bda01f8c31ebb72df80a46c88b25d1c79c56d"
uuid = "276daf66-3868-5448-9aa4-cd146d93841b"
version = "2.1.7"

[[deps.Statistics]]
deps = ["LinearAlgebra", "SparseArrays"]
uuid = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"

[[deps.TOML]]
deps = ["Dates"]
uuid = "fa267f1f-6049-4f14-aa54-33bafae1ed76"
version = "1.0.0"

[[deps.TableTraits]]
deps = ["IteratorInterfaceExtensions"]
git-tree-sha1 = "c06b2f539df1c6efa794486abfb6ed2022561a39"
uuid = "3783bdb8-4a98-5b6b-af9a-565f29a5fe9c"
version = "1.0.1"

[[deps.Tables]]
deps = ["DataAPI", "DataValueInterfaces", "IteratorInterfaceExtensions", "LinearAlgebra", "OrderedCollections", "TableTraits", "Test"]
git-tree-sha1 = "2d7164f7b8a066bcfa6224e67736ce0eb54aef5b"
uuid = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"
version = "1.9.0"

[[deps.Tar]]
deps = ["ArgTools", "SHA"]
uuid = "a4e569a6-e804-4fa4-b0f3-eef7a1d5b13e"
version = "1.10.0"

[[deps.TensorCore]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1feb45f88d133a655e001435632f019a9a1bcdb6"
uuid = "62fd8b95-f654-4bbd-a8a5-9c27f68ccd50"
version = "0.1.1"

[[deps.Test]]
deps = ["InteractiveUtils", "Logging", "Random", "Serialization"]
uuid = "8dfed614-e22c-5e08-85e1-65c5234f0b40"

[[deps.TranscodingStreams]]
deps = ["Random", "Test"]
git-tree-sha1 = "8a75929dcd3c38611db2f8d08546decb514fcadf"
uuid = "3bb67fe8-82b1-5028-8e26-92a6c54297fa"
version = "0.9.9"

[[deps.Tricks]]
git-tree-sha1 = "6bac775f2d42a611cdfcd1fb217ee719630c4175"
uuid = "410a4b4d-49e4-4fbc-ab6d-cb71b17b3775"
version = "0.1.6"

[[deps.UUIDs]]
deps = ["Random", "SHA"]
uuid = "cf7118a7-6976-5b1a-9a39-7adc72f591a4"

[[deps.UnPack]]
git-tree-sha1 = "387c1f73762231e86e0c9c5443ce3b4a0a9a0c2b"
uuid = "3a884ed6-31ef-47d7-9d2a-63182c4928ed"
version = "1.0.2"

[[deps.Unicode]]
uuid = "4ec0a83e-493e-50e2-b9ac-8f72acf5a8f5"

[[deps.WeakRefStrings]]
deps = ["DataAPI", "InlineStrings", "Parsers"]
git-tree-sha1 = "b1be2855ed9ed8eac54e5caff2afcdb442d52c23"
uuid = "ea10d353-3f73-51f8-a26c-33c1cb351aa5"
version = "1.4.2"

[[deps.Zlib_jll]]
deps = ["Libdl"]
uuid = "83775a58-1f1d-513f-b197-d71354ab007a"
version = "1.2.12+3"

[[deps.libblastrampoline_jll]]
deps = ["Artifacts", "Libdl", "OpenBLAS_jll"]
uuid = "8e850b90-86db-534c-a0d3-1478176c7d93"
version = "5.1.1+0"

[[deps.nghttp2_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850ede-7688-5339-a07c-302acd2aaf8d"
version = "1.48.0+0"

[[deps.p7zip_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "3f19e933-33d8-53b3-aaab-bd5110c3b7a0"
version = "17.4.0+0"
"""

# ╔═╡ Cell order:
# ╟─15112ba0-cddf-11eb-07a3-2bcd9257c8d0
# ╟─3cdf6437-0705-410c-8b0b-e8fc657897ed
# ╟─409f9114-3fe1-4abe-8beb-ab279e9031b2
# ╠═feb39810-13bf-4383-8184-9338111248cf
# ╟─6937b9af-1f25-4b10-8820-86f5db2f84e9
# ╟─ced63ea2-60c8-4913-a5d4-9e1b5ff47223
# ╟─6571612f-00ad-4179-bb93-a48cc9804352
# ╟─dac7fb80-10f7-4806-af0c-088637b79b69
# ╟─f1a259ac-e3c3-457a-9111-5eab394f452f
# ╟─0ba1252d-d891-4835-b81a-d63210f5c87b
# ╠═3ced63a3-0e7a-4a7a-ae26-5e4edae88cac
# ╟─6ea4d683-6825-4baa-bfe5-d8784f46420e
# ╠═f1acdec8-d55b-4b12-9b67-84c47be9378e
# ╟─ce89db42-eabe-4d30-8d0a-4b06ad04145a
# ╟─d38d5d7e-ef45-40e7-90c0-441fa4d30a24
# ╟─0120645b-0386-4fdd-b7fd-ace3f58c2d4d
# ╟─9690a57f-245b-4c2b-8ddc-0c9ef6b33ee8
# ╟─1b35a949-7c83-4d64-aa69-90b42c5dfbfb
# ╟─5235a517-8947-4d57-a436-c48eb3de87c6
# ╟─4c2622b9-191f-4e38-b5f4-c0eece8cb7af
# ╟─6bf1aae8-5d3a-4aa2-b01c-d633602f3c55
# ╟─0fd242c3-a99f-41b3-a5ed-1c92f6d3b600
# ╟─559be550-e0b0-4cd4-bd16-f7757bb190cb
# ╟─39562a1b-7a97-4ccd-b57c-9fb4bb15ba65
# ╟─85981e5d-1519-4dca-8739-4023e58a0176
# ╟─b9a1a968-24f3-4a92-8341-614bf5a7e0ca
# ╠═22e8290e-c39f-41e2-94e5-d9c3b1585d88
# ╟─3556cde6-e9b0-4dab-a8ae-dbc24f524b22
# ╟─ebb43969-4b22-4e8e-8694-6b47a8726b86
# ╟─894235bf-9515-4c82-b297-23e523764364
# ╟─4e65d2d8-c706-4a65-92ad-477e17ad39cb
# ╟─280f3cc5-0a2b-4cd3-bcfb-cba536877019
# ╟─644a26a3-0c5e-485c-ad2c-67afcc085d2d
# ╟─bb50e179-3e60-4fab-9be4-3b56f254c710
# ╟─5bbebe42-69cd-474a-ad1a-8599fed108be
# ╟─fd292c9b-ed3e-4f91-aea6-4142e429163a
# ╟─e28e76fd-14d5-4d32-8d98-f8aac4734c43
# ╟─7cab8ef9-0744-46c1-ab50-18c78ecbe931
# ╟─1fb6c731-ff80-41fd-b866-0db9ad4ec361
# ╟─51818620-e43c-4db4-813b-ed5c4562b62c
# ╟─5fe68e9d-3ff2-495e-b844-649e273ce758
# ╟─0f86d6fa-bdd4-4018-b3af-39e0bb1288f7
# ╟─6537fb39-c597-41ca-8b65-2d2912b0234a
# ╟─a1dc0363-8871-49d9-afc8-16a2660ae5c3
# ╟─4cc6e64e-4858-4f2a-b363-e7619c060bf8
# ╟─b99a4854-6671-4296-99b9-977af285812b
# ╟─e2bd0872-89de-4532-9be1-4b39c241133b
# ╟─51b3a17c-2454-4cff-a1cc-a2f27d047f91
# ╟─5ae9e073-7506-405e-a994-de7a08337bfa
# ╟─a71621ac-f4f6-4b54-8486-b7bdaa8d29ef
# ╟─747b89f4-fd33-4fc6-bd24-83b8860a5a5e
# ╟─b293a9cc-6c2e-4e8d-9445-9d377b83f676
# ╟─b8c26c04-3448-4e47-8fb6-9ba6f13ab210
# ╟─4ae438b1-0752-411f-ae55-52479f518c57
# ╟─d04e8abb-ae8e-4297-9734-533d9274274e
# ╟─94b9b913-2a22-4960-992a-699f18954be2
# ╟─cbc7414b-8788-4527-a766-abbc49512eaa
# ╟─7a82dcaf-b775-479a-94be-23a957ff0d9d
# ╟─97a132ae-a371-4449-a4bd-e2ce469cffca
# ╟─48a35af0-ab1d-4e6b-a195-0d3d885b5652
# ╟─8b8bdbd0-6148-4a9d-b2e1-d6625cb84ac4
# ╟─cef3ec74-5394-494c-af80-a29f469abe8e
# ╟─d4b320e8-b03a-4f44-b842-f7f6bc4724b9
# ╟─2c7c6904-52cd-459b-b459-daf6bdb4f91d
# ╟─40979b66-ab86-45ac-8102-64704d8bd613
# ╟─602a7f0f-6b1b-445c-8d3c-0887f858b1e0
# ╟─81c00071-63c2-48ff-96a9-b3ebcf8cc44f
# ╟─fd4a0888-c180-4f68-98e2-40061bc27354
# ╠═e0a35f3d-82d5-4315-ae83-27329529cf15
# ╠═18195c33-8d9e-431f-936b-b439a8b840a8
# ╟─31068d0a-0945-49a3-8c72-7a5ebb288db2
# ╟─1d1f906f-a672-4a0f-a35c-db1dfb0763fc
# ╟─89be032b-c88d-4de8-9f66-4acbdd3dec3d
# ╟─0d3c51a2-892b-4db1-a2af-fcddb75ef635
# ╟─1a055e0b-d23e-4d85-89bf-5813127e88ec
# ╟─4a4d51cb-8626-4d97-91aa-c4ce37b0a210
# ╟─f091ac64-be2c-4cc8-a7d3-34abde9fd5d4
# ╟─00000000-0000-0000-0000-000000000001
# ╟─00000000-0000-0000-0000-000000000002
